//
// (c) 2020 wesolutions GmbH
// All rights reserved.
//

import QtQuick 2.3

import wesual.Controls 1.0

FocusScope {
    id : textField

    property string placeholderText

    property alias inputMethodHints    : textInput.inputMethodHints

    property bool  multiline           : false
    property alias echoMode            : textInput.echoMode
    property alias border              : background.border
    property alias text                : textInput.text
    property alias textArea            : textEdit.text
    property alias validator           : textInput.validator
    property alias textColor           : textShown.color
    property alias font                : textShown.font
    property alias backgroundColor     : background.color
    property alias wrapMode            : textInput.wrapMode
    property alias horizontalAlignment : textInput.horizontalAlignment
    property alias verticalAlignment   : textInput.verticalAlignment
    property alias textFieldFontSize   : textInput.font.pixelSize
    property alias hovered             : mouseArea.containsMouse
    property alias acceptableInput     : textInput.acceptableInput

    readonly property alias impTextInputH : textInput.implicitHeight
    readonly property bool empty : {
        if (!multiline)
            return textInput.text.length === 0
        else
            return textEdit.text.length === 0
    }

    activeFocusOnTab : true

    Keys.onLeftPressed  : {}
    Keys.onRightPressed : {}
    Keys.onUpPressed    : {}
    Keys.onDownPressed  : {}

    signal editFinished()
    signal inputAccepted()

    QtObject {
        id : p_

        readonly property int   animationDuration : 0
    }

    implicitWidth  : 200
    implicitHeight : 26

    function selectAll() {
        textInput.selectAll();
    }

    function undoTextInput() {
        textInput.undo();
    }

    function setFocus() {
        if (multiline)
            textEdit.forceActiveFocus();
        else
            textInput.forceActiveFocus();
    }

    Rectangle {
        id : background
        color : UI.color(UI.PrimaryBase)
        border {
            color : {
                if (!enabled)
                    return UI.color(UI.DisabledBorder);

                if (textField.activeFocus || mouseArea.pressed)
                    return UI.color(UI.ControlBorderPress);
                else if (mouseArea.containsMouse || flickableMouseArea.containsMouse)
                    return UI.color(UI.ControlBorderHover);
                else
                    return UI.color(UI.PrimaryControlBorder);
            }
            width : 1
        }
        anchors.fill : parent
        Behavior on border.color {
            ColorAnimation { duration : p_.animationDuration }
        }
    }

    MouseArea {
        id : mouseArea
        anchors.fill : parent
        hoverEnabled : textField.enabled
        cursorShape  : Qt.IBeamCursor
        onDoubleClicked : selectAll()
        onClicked : {
            if (multiline)
                textEdit.forceActiveFocus();
            else
                textInput.forceActiveFocus();
        }
    }

    TextInput {
        id : textInput

        anchors.fill : parent
        anchors.margins : 4
        verticalAlignment : Qt.AlignVCenter
        selectedTextColor : UI.color(UI.SelectedTextColor)
        color : enabled ? UI.color(UI.PrimaryTextColor)
                        : UI.color(UI.DisabledText)
        selectionColor : UI.color(UI.TextSelection)
        font {
            family : UI.fontFamily(UI.PrimaryFont)
            weight : UI.fontWeight(UI.PrimaryFont)
            pixelSize : textShown.font.pixelSize
            italic : textShown.font.italic
        }
        activeFocusOnPress : true
        focus : visible ? true : false
        activeFocusOnTab : true
        passwordCharacter : "\u2022"
        selectByMouse : true
        clip : true
        visible : !multiline
        enabled : visible

        onEditingFinished : {
            textField.editFinished();
            //focus = false;
        }
        onAccepted        : {
            textField.inputAccepted();
            //focus = false;
        }
    }

    Flickable {
        id: flick

        anchors.fill  : parent
        anchors.margins : 4
        visible : multiline
        enabled : visible
        boundsBehavior : Flickable.StopAtBounds
        focus : visible
        contentHeight : textEdit.implicitHeight
        contentWidth  : textEdit.implicitWidth

        clip : true
        interactive : false
        onActiveFocusChanged : {
            if (activeFocus) {
                textEdit.focus = true;
            }
        }

        function ensureVisible(r)
        {
            if (contentX >= r.x)
                contentX = r.x;
            else if (contentX + width <= r.x + r.width)
                contentX = r.x + r.width - width;
         if (contentY >= r.y)
                contentY = r.y;
         else if (contentY + height <= r.y + r.height)
                contentY = r.y + r.height - height;
        }

        MouseArea {
           id : flickableMouseArea
           anchors.fill : parent
           hoverEnabled : textField.enabled
           cursorShape  : Qt.IBeamCursor
           onDoubleClicked : textEdit.selectAll()
           onClicked : textEdit.forceActiveFocus()

            TextEdit {
                id : textEdit

                width  : textField.width - 8
                height : Math.max(contentHeight, textField.height)
                selectedTextColor : UI.color(UI.SelectedTextColor)
                color : enabled ? UI.color(UI.PrimaryTextColor)
                                : UI.color(UI.DisabledText)
                selectionColor : UI.color(UI.TextSelection)
                onActiveFocusChanged : {
                    if (activeFocus)
                        selectAll();
                    else
                        textField.editFinished();
                }

                font {
                    family : UI.fontFamily(UI.PrimaryFont)
                    weight : UI.fontWeight(UI.PrimaryFont)
                    pixelSize : textShown.font.pixelSize
                    italic : textShown.font.italic
                }
                activeFocusOnPress : true
                activeFocusOnTab : true
                focus : true
                selectByMouse : true
                wrapMode : TextEdit.WordWrap

                Keys.onTabPressed : {
                    var next = nextItemInFocusChain(true);
                    if (next)
                        next.focus = true;
                }
                Keys.onBacktabPressed : {
                    var next = nextItemInFocusChain(false);
                    if (next)
                        next.focus = true;
                }
                onCursorRectangleChanged: flick.ensureVisible(cursorRectangle)
            }
        }
    }

    Text {
        id : textShown

        anchors.fill : parent
        anchors.margins : 4
        verticalAlignment : !multiline ? textInput.verticalAlignment :
                                         textEdit.verticalAlignment
        horizontalAlignment : !multiline ? textInput.horizontalAlignment :
                                           textEdit.horizontalAlignment
        font {
            family : UI.fontFamily(UI.PrimaryFont)
            weight : UI.fontWeight(UI.PrimaryFont)
            pixelSize : 14
        }
        text : {
            if (empty) {
                return textField.placeholderText || "";
            } else {
                if (!multiline)
                    return textInput.text;
                else
                    return textEdit.text;
            }
        }
        color : enabled ?
                UI.color(UI.SecondaryTextColor) : UI.color(UI.DisabledText)
        wrapMode : Text.NoWrap
        elide : Text.ElideRight
        opacity : {
            if (textInput.text.length !== 0 || textEdit.text.length !== 0) {
                return 0;
            } else if (textInput.activeFocus ||
                       textEdit.activeFocus  ||
                       !enabled) {
                return .6;
            } else {
                return 1;
            }
        }

        Behavior on opacity {
            NumberAnimation { duration : p_.animationDuration }
        }
    }

    focus : true
    Keys.onEnterPressed : {
        if (!multiline) {
            if (!textInput.acceptableInput) {
                textInput.undo();
            }
            focus = false;
        }
    }
    Keys.onReturnPressed : {
        if (!multiline) {
            if (!textInput.acceptableInput) {
                textInput.undo();
            }
            focus = false;
        }
    }
    Keys.onEscapePressed : {
        if (multiline)
            textEdit.undo();
        else
            textInput.undo();
        focus = false;
    }
}

